<?php

class Facturae {

    /* CONSTANTS */
    const SCHEMA_3_2_1 = "3.2.1";
    const SIGN_POLICY_3_1 = array(
    "name" => "Politica de Firma",
    "url" => "https://tribunet.hacienda.go.cr/docs/esquemas/2016/v4/Resolucion%20Comprobantes%20Electronicos%20%20DGT-R-48-2016.pdf",
    "digest" => "Ohixl6upD6av8N7pEvDABhEL6hM="
    );

    private $signTime = NULL;
    private $signPolicy = NULL;
    private $publicKey = NULL;
    private $privateKey = NULL;

    private $signatureID;
    private $signedInfoID;
    private $signedPropertiesID;
    private $signatureValueID;
    private $certificateID;
    private $referenceID;
    private $signatureSignedPropertiesID;
    private $signatureObjectID;

    /**
    * Generate random ID
    *
    * This method is used for generating random IDs required when signing the
    * document.
    *
    * @return int  Random number
    */
    private function random() {
        return rand(100000, 999999);
    }


    /**
    * Sign
    *
    * @param  string $path  Path to P12 file
    * @param  string $passphrase  Private key passphrase
    * @param  array  $policy      Facturae sign policy
    */
    public function sign($path, $passphrase, $policy=self::SIGN_POLICY_3_1) {
        openssl_pkcs12_read(file_get_contents($path), $fileCerts, $passphrase);
        $this->publicKey = $fileCerts['cert'];
        $this->privateKey = $fileCerts['pkey'];
        $this->privateKey = $fileCerts['pkey'];
        $this->signPolicy = $policy;

        // Generate random IDs
        $this->signatureID = $this->random();
        $this->signedInfoID = $this->random();
        $this->signedPropertiesID = $this->random();
        $this->signatureValueID = $this->random();
        $this->certificateID = $this->random();
        $this->referenceID = $this->random();
        $this->signatureSignedPropertiesID = $this->random();
        $this->signatureObjectID = $this->random();
    }


    /**
    * Inject signature
    * @param  string Unsigned XML document
    * @return string Signed XML document
    */
    private function injectSignature($xml) {
        // Make sure we have all we need to sign the document
        if (is_null($this->publicKey) || is_null($this->privateKey)) return $xml;

        // Normalize document
        $xml = str_replace("\r", "", $xml);

        // Define namespace
        $xmlns = 'xmlns:ds="http://www.w3.org/2000/09/xmldsig#" ' .
            'xmlns:etsi="http://uri.etsi.org/01903/v1.3.2#" ' .
            'xmlns="https://tribunet.hacienda.go.cr/docs/esquemas/2016/v4.1/facturaElectronica"';;

        // Prepare signed properties
        $signTime = is_null($this->signTime) ? time() : $this->signTime;
        $certData = openssl_x509_parse($this->publicKey);
        $certDigest = openssl_x509_fingerprint($this->publicKey, "sha1", true);
        $certDigest = base64_encode($certDigest);
        $certIssuer = "CN=" . $certData['issuer']['CN'] . "," .
                      "OU=" . $certData['issuer']['OU'] . "," .
                      "O=" .  $certData['issuer']['O']  . "," .
                      "C=" .  $certData['issuer']['C'];
        $prop = '<etsi:SignedProperties Id="Signature' . $this->signatureID .
          '-SignedProperties' . $this->signatureSignedPropertiesID . '">' .
          '<etsi:SignedSignatureProperties><etsi:SigningTime>' .
          date('c', $signTime) . '</etsi:SigningTime>' .
          '<etsi:SigningCertificate><etsi:Cert><etsi:CertDigest>' .
          '<ds:DigestMethod Algorithm="http://www.w3.org/2000/09/xmldsig#sha1">' .
          '</ds:DigestMethod><ds:DigestValue>' . $certDigest .
          '</ds:DigestValue></etsi:CertDigest><etsi:IssuerSerial>' .
          '<ds:X509IssuerName>' . $certIssuer .
          '</ds:X509IssuerName><ds:X509SerialNumber>' .
          $certData['serialNumber'] . '</ds:X509SerialNumber>' .
          '</etsi:IssuerSerial></etsi:Cert></etsi:SigningCertificate>' .
          '<etsi:SignaturePolicyIdentifier><etsi:SignaturePolicyId>' .
          '<etsi:SigPolicyId><etsi:Identifier>' . $this->signPolicy['url'] .
          '</etsi:Identifier><etsi:Description>' . $this->signPolicy['name'] .
          '</etsi:Description></etsi:SigPolicyId><etsi:SigPolicyHash>' .
          '<ds:DigestMethod Algorithm="http://www.w3.org/2000/09/xmldsig#sha1">' .
          '</ds:DigestMethod><ds:DigestValue>' . $this->signPolicy['digest'] .
          '</ds:DigestValue></etsi:SigPolicyHash></etsi:SignaturePolicyId>' .
          '</etsi:SignaturePolicyIdentifier><etsi:SignerRole>' .
          '<etsi:ClaimedRoles><etsi:ClaimedRole>emisor</etsi:ClaimedRole>' .
          '</etsi:ClaimedRoles></etsi:SignerRole>' .
          '</etsi:SignedSignatureProperties><etsi:SignedDataObjectProperties>' .
          '<etsi:DataObjectFormat ObjectReference="#Reference-ID-' .
          $this->referenceID . '"><etsi:Description>Factura electronica' .
          '</etsi:Description><etsi:MimeType>text/xml</etsi:MimeType>' .
          '</etsi:DataObjectFormat></etsi:SignedDataObjectProperties>' .
          '</etsi:SignedProperties>';

        // Prepare key info
        $kInfo = '<ds:KeyInfo Id="Certificate' . $this->certificateID . '">' .
          "\n" . '<ds:X509Data>' . "\n" . '<ds:X509Certificate>' . "\n";
        $publicPEM = "";
        openssl_x509_export($this->publicKey, $publicPEM);
        $publicPEM = str_replace("-----BEGIN CERTIFICATE-----", "", $publicPEM);
        $publicPEM = str_replace("-----END CERTIFICATE-----", "", $publicPEM);
        $publicPEM = str_replace("\n", "", $publicPEM);
        $publicPEM = str_replace("\r", "", chunk_split($publicPEM, 76));
        $kInfo .= $publicPEM . '</ds:X509Certificate>' . "\n" . '</ds:X509Data>' .
          "\n" . '</ds:KeyInfo>';

        // Calculate digests
        $propDigest = base64_encode(sha1(str_replace('<etsi:SignedProperties',
          '<etsi:SignedProperties ' . $xmlns, $prop), true));
        $kInfoDigest = base64_encode(sha1(str_replace('<ds:KeyInfo',
          '<ds:KeyInfo ' . $xmlns, $kInfo), true));
        $documentDigest = base64_encode(sha1($xml, true));

        // Prepare signed info
        $sInfo = '<ds:SignedInfo Id="Signature-SignedInfo' . $this->signedInfoID .
            '">' . "\n" . '<ds:CanonicalizationMethod Algorithm="' .
            'http://www.w3.org/TR/2001/REC-xml-c14n-20010315">' .
            '</ds:CanonicalizationMethod>' . "\n" . '<ds:SignatureMethod ' .
            'Algorithm="http://www.w3.org/2000/09/xmldsig#rsa-sha1">' .
            '</ds:SignatureMethod>' . "\n" . '<ds:Reference Id="SignedPropertiesID' .
            $this->signedPropertiesID . '" ' .
            'Type="http://uri.etsi.org/01903#SignedProperties" ' .
            'URI="#Signature' . $this->signatureID . '-SignedProperties' .
            $this->signatureSignedPropertiesID . '">' . "\n" . '<ds:DigestMethod ' .
            'Algorithm="http://www.w3.org/2000/09/xmldsig#sha1">' .
            '</ds:DigestMethod>' . "\n" . '<ds:DigestValue>' . $propDigest .
            '</ds:DigestValue>' . "\n" . '</ds:Reference>' . "\n" . '<ds:Reference ' .
            'URI="#Certificate' . $this->certificateID . '">' . "\n" .
            '<ds:DigestMethod Algorithm="http://www.w3.org/2000/09/xmldsig#sha1">' .
            '</ds:DigestMethod>' . "\n" . '<ds:DigestValue>' . $kInfoDigest .
            '</ds:DigestValue>' . "\n" . '</ds:Reference>' . "\n" .
            '<ds:Reference Id="Reference-ID-' . $this->referenceID . '" URI="">' .
            "\n" . '<ds:Transforms>' . "\n" . '<ds:Transform Algorithm="' .
            'http://www.w3.org/2000/09/xmldsig#enveloped-signature">' .
            '</ds:Transform>' . "\n" . '</ds:Transforms>' . "\n" .
            '<ds:DigestMethod Algorithm="http://www.w3.org/2000/09/xmldsig#sha1">' .
            '</ds:DigestMethod>' . "\n" . '<ds:DigestValue>' . $documentDigest .
            '</ds:DigestValue>' . "\n" . '</ds:Reference>' . "\n" .'</ds:SignedInfo>';

        // Calculate signature
        $signaturePayload = str_replace('<ds:SignedInfo',
          '<ds:SignedInfo ' . $xmlns, $sInfo);
        $signatureResult = "";
        openssl_sign($signaturePayload, $signatureResult, $this->privateKey);
        $signatureResult = chunk_split(base64_encode($signatureResult), 76);
        $signatureResult = str_replace("\r", "", $signatureResult);

        // Make signature
        $sig = '<ds:Signature xmlns:etsi="http://uri.etsi.org/01903/v1.3.2#" ' .
          'Id="Signature' . $this->signatureID . '">' . "\n" . $sInfo . "\n" .
          '<ds:SignatureValue Id="SignatureValue' . $this->signatureValueID . '">' .
          "\n" . $signatureResult . '</ds:SignatureValue>' . "\n" . $kInfo . "\n" .
          '<ds:Object Id="Signature' . $this->signatureID . '-Object' .
          $this->signatureObjectID . '"><etsi:QualifyingProperties ' .
          'Target="#Signature' . $this->signatureID . '">' . $prop .
          '</etsi:QualifyingProperties></ds:Object></ds:Signature>';

        // Inject signature
        $xml = str_replace('</FacturaElectronica>', $sig . '</FacturaElectronica>', $xml);

        return $xml;
    }

    /**
    * Export
    *
    * Get Facturae XML data
    *
    * @param  string     $xml Path to save invoice
    * @param  string     $filePath Path to save invoice
    * @return string|int           XML data|Written file bytes
    */
    public function export($xml, $filePath=NULL) {

        // Add signature
        $xml = $this->injectSignature($xml);

        // Prepend content type
        $xml = '<?xml version="1.0" encoding="UTF-8"?>' . "\n" . $xml;

        // Save document
        if (!is_null($filePath)) {
            file_put_contents('base64.txt', base64_encode($xml));
            return file_put_contents($filePath, $xml);
        }

        return $xml;
    }
}
